/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.ipt.config;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Random;
import java.util.stream.Stream;
import javax.validation.constraints.NotNull;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.gbif.ipt.config.DataDir;
import org.gbif.ipt.model.FileSource;
import org.gbif.ipt.model.Resource;
import org.gbif.ipt.service.InvalidConfigException;
import org.gbif.ipt.utils.InputStreamUtils;

public class DataDir {
    public static final String LOGGING_DIR = "logs";
    public static final String CONFIG_DIR = "config";
    public static final String RESOURCES_DIR = "resources";
    public static final String TMP_DIR = "tmp";
    public static final String PERSISTENCE_FILENAME = "resource.xml";
    public static final String INFERRED_METADATA_FILENAME = "inferredMetadata.xml";
    public static final String EML_XML_FILENAME = "eml.xml";
    public static final String FRICTIONLESS_METADATA_FILENAME = "datapackage.json";
    public static final String COL_DP_METADATA_FILENAME = "metadata.yaml";
    public static final String DWCA_FILENAME = "dwca.zip";
    public static final String PUBLICATION_LOG_FILENAME = "publication.log";
    private static final Random RANDOM = new Random();
    private static final Logger LOG = LogManager.getLogger(DataDir.class);
    protected File dataDir;
    private File dataDirSettingFile;
    private InputStreamUtils streamUtils = new InputStreamUtils();

    private DataDir() {
    }

    public static DataDir buildFromLocationFile(File dataDirSettingFile) {
        DataDir dd = new DataDir();
        dd.dataDirSettingFile = dataDirSettingFile;
        if (dataDirSettingFile != null && dataDirSettingFile.exists()) {
            String dataDirPath = null;
            try {
                dataDirPath = StringUtils.trimToNull((String)FileUtils.readFileToString((File)dataDirSettingFile, (String)"UTF-8"));
                if (dataDirPath != null) {
                    LOG.info("IPT Data Directory configured at " + dataDirPath);
                    dd.dataDir = new File(dataDirPath);
                }
            }
            catch (IOException e) {
                LOG.error("Failed to read the IPT Data Directory location settings file in WEB-INF at " + dataDirSettingFile.getAbsolutePath(), (Throwable)e);
            }
        } else {
            LOG.warn("IPT Data Directory location settings file in WEB-INF not found. Continue without data directory.");
        }
        return dd;
    }

    public static DataDir buildFromString(String dataDirPath) {
        DataDir dd = new DataDir();
        if (dataDirPath != null) {
            LOG.info("IPT Data Directory configured at " + dataDirPath);
            dd.dataDir = new File(dataDirPath);
        }
        return dd;
    }

    private void assureDirExists(File f) {
        if (f != null && !f.exists()) {
            f.mkdirs();
        }
    }

    private void assureParentExists(File f) {
        if (f != null && !f.getParentFile().exists()) {
            f.getParentFile().mkdirs();
        }
    }

    protected void clearTmp() throws IOException {
        File tmpDir = this.tmpFile("");
        FileUtils.forceMkdir((File)tmpDir);
        FileUtils.cleanDirectory((File)tmpDir);
        LOG.debug("Cleared temporary folder");
    }

    public File configFile(String path) {
        return this.dataFile("config/" + path);
    }

    private void createDefaultDir() throws IOException {
        File configDir = this.dataFile(CONFIG_DIR);
        File resourcesDir = this.dataFile(RESOURCES_DIR);
        File loggingDir = this.dataFile(LOGGING_DIR);
        FileUtils.forceMkdir((File)configDir);
        FileUtils.forceMkdir((File)resourcesDir);
        FileUtils.forceMkdir((File)loggingDir);
        InputStream input = this.streamUtils.classpathStream("configDefault/ipt.properties");
        if (input == null) {
            throw new InvalidConfigException(InvalidConfigException.TYPE.CONFIG_WRITE, "Cannot read required classpath resources to create new data dir!");
        }
        org.gbif.ipt.utils.FileUtils.copyStreamToFile((InputStream)input, (File)this.configFile("ipt.properties"));
        LOG.info("Creating new default data dir");
    }

    public File dataFile(String path) {
        if (this.dataDir == null) {
            throw new IllegalStateException("No data dir has been configured yet");
        }
        File f = new File(this.dataDir, path);
        this.assureParentExists(f);
        return f;
    }

    public boolean isConfigured() {
        return this.dataDir != null && this.dataDir.isDirectory() && this.dataDir.list().length > 0;
    }

    public boolean isConfiguredButEmpty() {
        return this.dataDir != null && this.dataDir.isDirectory() && this.dataDir.list().length == 0;
    }

    public File configDir() {
        return this.dataFile(CONFIG_DIR);
    }

    public File tmpRootDir() {
        return this.dataFile(TMP_DIR);
    }

    public File resourcesDir() {
        return this.dataFile(RESOURCES_DIR);
    }

    public File loggingDir() {
        return this.dataFile(LOGGING_DIR);
    }

    public File loggingFile(String path) {
        return this.dataFile("logs/" + path);
    }

    private void persistLocation() throws IOException {
        FileUtils.writeStringToFile((File)this.dataDirSettingFile, (String)this.dataDir.getAbsolutePath(), (Charset)StandardCharsets.UTF_8);
        LOG.info("IPT DataDir location file in /WEB-INF changed to " + this.dataDir.getAbsolutePath());
    }

    public File resourceDwcaFile(@NotNull String resourceName, @NotNull BigDecimal version) {
        String fn = "dwca-" + version.toPlainString() + ".zip";
        return this.dataFile("resources/" + resourceName + "/" + fn);
    }

    public File resourceArchiveFile(@NotNull Resource resource, @NotNull BigDecimal version) {
        if (resource.isDataPackage()) {
            return this.resourceDataPackageFile(resource.getShortname(), version);
        }
        return this.resourceDwcaFile(resource.getShortname(), version);
    }

    public File resourceDataPackageFile(@NotNull String resourceName, @NotNull BigDecimal version) {
        String fn = "datapackage-" + version.toPlainString() + ".zip";
        return this.dataFile("resources/" + resourceName + "/" + fn);
    }

    public File resourceDwcaFile(@NotNull String resourceName) {
        return this.dataFile("resources/" + resourceName + "/dwca.zip");
    }

    public File resourceEmlFile(@NotNull String resourceName, @NotNull BigDecimal version) {
        String fn = "eml-" + version.toPlainString() + ".xml";
        return this.dataFile("resources/" + resourceName + "/" + fn);
    }

    public File resourceDatapackageMetadataFile(@NotNull String resourceName, String type, @NotNull BigDecimal version) {
        if ("coldp".equals(type)) {
            String fn = "metadata-" + version.toPlainString() + ".yaml";
            return this.dataFile("resources/" + resourceName + "/" + fn);
        }
        String fn = "datapackage-" + version.toPlainString() + ".json";
        return this.dataFile("resources/" + resourceName + "/" + fn);
    }

    public File resourceEmlFile(@NotNull String resourceName) {
        return this.dataFile("resources/" + resourceName + "/eml.xml");
    }

    public File resourceDatapackageMetadataFile(@NotNull String resourceName, String type) {
        if ("coldp".equals(type)) {
            return this.dataFile("resources/" + resourceName + "/metadata.yaml");
        }
        return this.dataFile("resources/" + resourceName + "/datapackage.json");
    }

    public File resourceFile(Resource resource, String path) {
        if (resource == null) {
            return null;
        }
        return this.resourceFile(resource.getShortname(), path);
    }

    public File resourceFile(String resourceName, String path) {
        return this.dataFile("resources/" + resourceName + "/" + path);
    }

    public File resourceFile(String resourceName) {
        return this.dataFile("resources/" + resourceName + "/resource.xml");
    }

    public File resourceFile(Resource resource) {
        return resource == null ? null : this.dataFile("resources/" + resource.getShortname() + "/resource.xml");
    }

    public File resourceInferredMetadataFile(String resourceName) {
        return this.dataFile("resources/" + resourceName + "/inferredMetadata.xml");
    }

    public File resourceLogoFile(String resourceName, String suffix) {
        if (suffix == null) {
            suffix = "jpeg";
        }
        suffix = suffix.toLowerCase();
        return this.dataFile("resources/" + resourceName + "/logo." + suffix);
    }

    public File appLogoFile(String suffix) {
        if (suffix == null) {
            suffix = "jpeg";
        }
        suffix = suffix.toLowerCase();
        return this.dataFile("config/.uiSettings/logos/logo." + suffix);
    }

    public void removeLogoFile() {
        File logosDirectory = new File(this.dataDir, "config/.uiSettings/logos");
        File[] logoFiles = logosDirectory.listFiles();
        if (logoFiles != null) {
            Stream.of(logoFiles).filter(file -> !file.isDirectory()).filter(file -> file.getName().startsWith("logo")).forEach(File::delete);
        }
    }

    public File resourcePublicationLogFile(String resourceName) {
        return this.dataFile("resources/" + resourceName + "/publication.log");
    }

    public File resourceRtfFile(@NotNull String resourceName, @NotNull BigDecimal version) {
        String fn = resourceName + "-" + version.toPlainString() + ".rtf";
        return this.dataFile("resources/" + resourceName + "/" + fn);
    }

    public boolean setDataDir(File dataDir) throws InvalidConfigException {
        if (dataDir == null) {
            throw new NullPointerException("DataDir file required");
        }
        this.dataDir = dataDir;
        File configDir = this.configFile("");
        if (dataDir.exists() && (!dataDir.isDirectory() || dataDir.list().length > 0)) {
            if (dataDir.isDirectory()) {
                if (!configDir.exists() || !configDir.isDirectory()) {
                    this.dataDir = null;
                    throw new InvalidConfigException(InvalidConfigException.TYPE.INVALID_DATA_DIR, "DataDir " + dataDir.getAbsolutePath() + " exists already and is no IPT data dir.");
                }
                LOG.info("Reusing existing data dir {}", (Object)dataDir);
                try {
                    this.persistLocation();
                }
                catch (IOException e) {
                    LOG.error("Cant persist datadir location in WEBINF webapp folder", (Throwable)e);
                }
                return false;
            }
            this.dataDir = null;
            throw new InvalidConfigException(InvalidConfigException.TYPE.INVALID_DATA_DIR, "DataDir " + dataDir.getAbsolutePath() + " is not a directory");
        }
        LOG.info("Setting up new data directory {}", (Object)dataDir);
        try {
            FileUtils.forceMkdir((File)dataDir);
            File testFile = new File(dataDir, "test.tmp");
            FileUtils.touch((File)testFile);
            testFile.delete();
            this.createDefaultDir();
            if (this.dataDirSettingFile != null) {
                this.persistLocation();
            }
            return true;
        }
        catch (IOException e) {
            LOG.error("New DataDir " + dataDir.getAbsolutePath() + " not writable", (Throwable)e);
            this.dataDir = null;
            throw new InvalidConfigException(InvalidConfigException.TYPE.NON_WRITABLE_DATA_DIR, "DataDir " + dataDir.getAbsolutePath() + " is not writable");
        }
    }

    public File sourceFile(Resource resource, FileSource source) {
        if (resource == null) {
            return null;
        }
        return this.resourceFile(resource.getShortname(), "sources/" + source.getName() + source.getPreferredFileSuffix());
    }

    public File sourceFile(Resource resource, String filename) {
        if (resource == null) {
            return null;
        }
        return this.resourceFile(resource.getShortname(), "sources/" + filename);
    }

    public File sourceLogFile(String resourceName, String sourceName) {
        return this.dataFile("resources/" + resourceName + "/sources/" + sourceName + ".log");
    }

    public File tmpDir() {
        String random = String.valueOf(RANDOM.nextLong());
        File dir = this.tmpFile("dir" + random);
        this.assureDirExists(dir);
        return dir;
    }

    public File tmpFile(String path) {
        return this.dataFile("tmp/" + path);
    }

    public File tmpFile(String prefix, String suffix) {
        String random = String.valueOf(RANDOM.nextInt());
        return this.tmpFile(prefix + random + suffix);
    }

    public long getDataDirTotalSpace() {
        return this.dataDir.getTotalSpace();
    }

    public long getDataDirUsableSpace() {
        return this.dataDir.getUsableSpace();
    }

    public DirStatus getDirectoryReadWriteStatus(File dir) {
        if (dir == null || !dir.exists()) {
            return DirStatus.NOT_EXIST;
        }
        if (!dir.canRead()) {
            return DirStatus.NO_ACCESS;
        }
        if (dir.canRead() && !dir.canWrite()) {
            return DirStatus.READ_ONLY;
        }
        return DirStatus.READ_WRITE;
    }

    public DirStatus getSubDirectoriesReadWriteStatus(File dir) {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File subDir : files) {
                DirStatus status = this.getDirectoryReadWriteStatus(subDir);
                if (status == DirStatus.READ_WRITE) continue;
                return status;
            }
            return DirStatus.READ_WRITE;
        }
        return DirStatus.NOT_EXIST;
    }

    public File getDataDir() {
        return this.dataDir;
    }
}

